#!/bin/bash

# Script to pull latest changes from origin/master for all modules
# Usage: ./pull_modules.sh

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Get the script directory
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
MODULES_DIR="$SCRIPT_DIR/Modules"

# Counters for stats
TOTAL_MODULES=0
SUCCESSFUL_PULLS=0
FAILED_PULLS=0
NOT_GIT_REPOS=0

# Arrays to store results
SUCCESSFUL_MODULES=()
FAILED_MODULES=()
NOT_GIT_MODULES=()

echo -e "${BLUE}========================================${NC}"
echo -e "${BLUE}  Module Git Pull Script${NC}"
echo -e "${BLUE}========================================${NC}"
echo ""

# Check if Modules directory exists
if [ ! -d "$MODULES_DIR" ]; then
    echo -e "${RED}Error: Modules directory not found at $MODULES_DIR${NC}"
    exit 1
fi

# Iterate through each module directory
for module_dir in "$MODULES_DIR"/*; do
    # Check if it's a directory
    if [ -d "$module_dir" ]; then
        module_name=$(basename "$module_dir")
        TOTAL_MODULES=$((TOTAL_MODULES + 1))

        echo -e "${YELLOW}Processing: $module_name${NC}"

        # Check if it's a git repository
        if [ -d "$module_dir/.git" ]; then
            # Change to module directory
            cd "$module_dir" || continue

            # Pull from origin/master
            echo -e "  ${BLUE}Running: git pull origin master${NC}"
            if git pull origin master 2>&1; then
                echo -e "  ${GREEN}✓ Successfully pulled $module_name${NC}"
                SUCCESSFUL_PULLS=$((SUCCESSFUL_PULLS + 1))
                SUCCESSFUL_MODULES+=("$module_name")
            else
                echo -e "  ${RED}✗ Failed to pull $module_name${NC}"
                FAILED_PULLS=$((FAILED_PULLS + 1))
                FAILED_MODULES+=("$module_name")
            fi

            # Return to script directory
            cd "$SCRIPT_DIR" || exit 1
        else
            echo -e "  ${YELLOW}⚠ Not a git repository (skipping)${NC}"
            NOT_GIT_REPOS=$((NOT_GIT_REPOS + 1))
            NOT_GIT_MODULES+=("$module_name")
        fi

        echo ""
    fi
done

# Print statistics
echo -e "${BLUE}========================================${NC}"
echo -e "${BLUE}  Statistics Summary${NC}"
echo -e "${BLUE}========================================${NC}"
echo ""
echo -e "Total Modules Processed: ${BLUE}$TOTAL_MODULES${NC}"
echo -e "Successful Pulls: ${GREEN}$SUCCESSFUL_PULLS${NC}"
echo -e "Failed Pulls: ${RED}$FAILED_PULLS${NC}"
echo -e "Not Git Repositories: ${YELLOW}$NOT_GIT_REPOS${NC}"
echo ""

# Print successful modules
if [ ${#SUCCESSFUL_MODULES[@]} -gt 0 ]; then
    echo -e "${GREEN}✓ Successfully Pulled Modules ($SUCCESSFUL_PULLS):${NC}"
    for i in "${!SUCCESSFUL_MODULES[@]}"; do
        echo -e "  ${GREEN}  $((i + 1)). ${SUCCESSFUL_MODULES[$i]}${NC}"
    done
    echo ""
fi

# Print failed modules
if [ ${#FAILED_MODULES[@]} -gt 0 ]; then
    echo -e "${RED}✗ Failed Modules ($FAILED_PULLS):${NC}"
    for i in "${!FAILED_MODULES[@]}"; do
        echo -e "  ${RED}  $((i + 1)). ${FAILED_MODULES[$i]}${NC}"
    done
    echo ""
fi

# Print non-git modules
if [ ${#NOT_GIT_MODULES[@]} -gt 0 ]; then
    echo -e "${YELLOW}⚠ Non-Git Modules (Skipped) ($NOT_GIT_REPOS):${NC}"
    for i in "${!NOT_GIT_MODULES[@]}"; do
        echo -e "  ${YELLOW}  $((i + 1)). ${NOT_GIT_MODULES[$i]}${NC}"
    done
    echo ""
fi

# Exit with appropriate code
if [ $FAILED_PULLS -gt 0 ]; then
    echo -e "${RED}Script completed with errors${NC}"
    exit 1
else
    echo -e "${GREEN}Script completed successfully${NC}"
    exit 0
fi
