#!/bin/sh

read_var() {
    if [ -z "$1" ]; then
        echo "environment variable name is required"
        return
    fi

    local ENV_FILE='.env'
    if [ ! -z "$2" ]; then
        ENV_FILE="$2"
    fi

    local VAR=$(grep $1 "$ENV_FILE" | xargs)
    IFS="=" read -ra VAR <<<"$VAR"
    echo ${VAR[1]}
}

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

printf "${BLUE}-> Reading environment variables...${NC}\n"
autoFolder=$(read_var AUTO_UPDATE_FOLDER)
product=$(read_var APP_PRODUCT)

printf "${BLUE}-> Setting up file names and paths...${NC}\n"
updaterFilename="laraupdater.json"
version=$(<public/version.txt)
filename="$product-auto-$version"
autoUpdateFolder="$autoFolder/"$product

printf "${GREEN}-> Current version: ${NC}$version\n"
printf "${GREEN}-> Auto update folder: ${NC}$autoUpdateFolder\n"

logs=()
printf "\n${YELLOW}==========================================\n"
printf "Enter Change Log for version $version\n"
printf "Write 'END' when you're finished\n"
printf "==========================================${NC}\n\n"

while read log; do
    if [ "$log" == "END" ]; then
        break
    fi
    logs+=("<li>"$log"</li>")
done

printf "${BLUE}-> Generating $updaterFilename file...${NC}\n"
now="$(date +'%Y-%m-%d')"

echo "{
    \"version\": \"$version\",
    \"archive\": \"$filename.zip\",
    \"description\": \"<ul>${logs[@]}</ul>\",
    \"release_date\": \"${now}\"
}"

echo "{
    \"version\": \"$version\",
    \"archive\": \"$filename.zip\",
    \"description\": \"<ul>${logs[@]}</ul>\",
    \"release_date\": \"${now}\"
}" >$updaterFilename

printf "${BLUE}-> Deleting old $filename.zip file...${NC}\n"
rm -rf "$autoUpdateFolder/$filename".zip

printf "${BLUE}-> Zipping files...${NC}\n"

# shellcheck disable=SC2164
cd "$autoUpdateFolder" && zip -r "$filename".zip . && cd -
printf "${GREEN}✓ Zipping completed successfully${NC}\n"

while true; do
    printf "\n${YELLOW}==========================================\n"
    # shellcheck disable=SC2039
    read -p "Do you want to upload $filename.zip and $updaterFilename to S3 server? (y/n) " yn
    printf "==========================================${NC}\n"
    case $yn in
    [Yy]*)
        printf "${BLUE}-> Starting S3 upload process...${NC}\n"
        printf "${BLUE}-> Uploading $filename.zip to froiden-update-hub/$product/...${NC}\n"
        aws s3 cp "$autoUpdateFolder/$filename".zip s3://froiden-update-hub"/$product/"

        printf "${BLUE}-> Uploading $updaterFilename with public read access...${NC}\n"
        aws s3 cp $updaterFilename s3://froiden-update-hub"/$product/" --grants read=uri=http://acs.amazonaws.com/groups/global/AllUsers
        printf "${GREEN}✓ S3 upload completed successfully${NC}\n"
        break
        ;;
    [Nn]*)
        printf "${YELLOW}-> Upload cancelled${NC}\n"
        break
        ;;
    *) printf "${RED}Please answer 'y' for yes or 'n' for no.${NC}\n" ;;
    esac
done

printf "\n${GREEN}=========================================================\n"
printf "                 UPLOAD PROCESS COMPLETE                   \n"
printf "=========================================================\n${NC}"
printf "${BLUE}-> Version: ${NC}$version\n"
printf "${BLUE}-> Files uploaded:${NC}\n"
printf "   - $filename.zip\n"
printf "   - $updaterFilename\n"
printf "${BLUE}-> Location: ${NC}s3://froiden-update-hub/$product/\n"
printf "${GREEN}=========================================================\n${NC}\n"
